;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  clock  ;; the number of times thru the go procedure
  speed-index  ;; the index for speed-list which gives the current speed of each of the turtles
  max-speed  ;; the maximum speed that any molecule will have in a particular run
  num-speeds  ;; the number of speeds for the molecules that were collected

;; quick start instructions variables
  quick-start  ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item  ;; the index of the current quickstart instruction
  qs-items  ;; the list of quickstart instructions

  speed-frequency-sum-list  ;; the list of all the speeds all the turtles have moved thru now
]

turtles-own
[
  speed  ;; the speed of this turtle at this time
  user-id  ;; the user-id of the calculator associated with this turtle
  speed-list  ;; the list of all the speeds for this turtle
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup
  setup-quick-start
  hubnet-set-client-interface "TI-83+" [ "AAA - People Molecules 1.1" [ "L1" ] ]
  hubnet-reset
end

;; Initializes the display, the plot, and the global variables.
to setup
  clear-output
  cp
  ct
  clear-all-plots
  
  set clock 0
  set speed-index 0
  set max-speed 0
  set num-speeds 0
  set speed-frequency-sum-list []

  set-default-shape turtles "circle"

  ask patches with [abs pxcor = screen-edge-x or abs pycor = screen-edge-y]
  [ set pcolor yellow ]

  setup-histograms
end

;; setups the histogram to have the right pen color, etc.
to setup-histograms
  set-current-plot "Speed Histogram"
  clear-plot
  set-histogram-num-bars 20  ;; set the number of bars to be the same as in the calc's histograms
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "clear the calulator memory. (Press 2nd MEM 7 1 2)"
                 + "\n  -Open the teacher console for this activity and press the ERASE ALL DATA button."
    setup
  ]
end

;; only used in testing- creates some turtles to test the activity
to make-test-turtles
  cct 30
  [ init-test-turtles ]
  set-color
  ask random-one-of turtles
  [ set num-speeds length speed-list ]
  set max-speed max values-from turtles [ max speed-list ]
end

;; only used in testing- intializes the turtle's variables
to init-test-turtles  ;; turtle procedure
  locals
  [ index ]
  
  set user-id -1
  setxy ((random-float (2 * screen-edge-x - 1)) - screen-edge-x + .5) ((random-float (2 * screen-edge-y - 1)) - screen-edge-y + .5)
  rt random-float 360
  set index 0
  set speed 0
  set speed-list []
  repeat 20
  [
    set speed-list (lput (random-float 5.0) speed-list)
    set index (index + 1)
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; receives information from the calculators and runs the simulation
to go
  ;; stop running the simulation if the there are no more speeds in the speed-list
  if speed-index > num-speeds
  [ stop ]

  ;; every time clock is divisible by histogram-display-interval get a new speed and plot the histogram again
  if clock mod histogram-display-interval = 0
  [
    update-speed
    set-color
    do-histograms
  ]
  ;; make the turtles move forward by their current speed
  ask turtles
  [ move ] 
  if max values-from turtles [ speed ] < 1
  [ wait delay ]

  set clock (clock + 1)
end

;; get the next speed from the speed-list, if there is one
to update-speed
  if speed-index < num-speeds
  [
    ask turtles
    [ set speed (item speed-index speed-list) ]
  ]

  set speed-index (speed-index + 1)
end

;; color the turtles based on their speed, make them blue if they are slow, green if they have a medium speed, and red if they are fast
;; note: this range is based on the highest speed in all the turtles speed-list
to set-color
  ask turtles
  [
    ifelse speed <= (max-speed / 3)
    [ set color blue ]
    [
      ifelse speed > (max-speed / 3) and speed <= (2 * max-speed / 3)
      [ set color green ]
      [ set color red ]
    ]
  ]
end

;; move the turtles forward by their current speed
to move  ;; turtle procedure
  locals
  [
    remaining-distance
  ]

  set remaining-distance speed

  ;; while the remaining distance to travel is not 0, move the turtle forward by 1 or the remaining amount if it is less than 1
  while [ remaining-distance > 0 ]
  [
    ;; slow the movement down so that it can be seen by the user and scale it so that the fast molecules move quickly and slow molecules move slowly
    every delay * ( max-speed / speed ) / 10
    [
      ;; change the heading of the turtle if it is hitting a wall
      if abs pxcor-of patch-ahead .1 >= screen-edge-x
      [ set heading (360 - heading) ]
      if abs pycor-of patch-ahead .1 >= screen-edge-y
      [ set heading (180 - heading) ]

      ;; move the turtle forward by 1 if remaining-distance is greater than 1
      ifelse remaining-distance > .1
      [ fd .1 ]
      [ fd remaining-distance ]

      ;; decrement remaining-distance
      set remaining-distance (remaining-distance - .1)

      ;; if there are other turtles here, give this turtle a new direction since the molecules hit each other
      if any? other-turtles-here
      [ rt random-float 360 ]
    ]
  ]
end

;; reports the amount of seconds by which to slow the model down
to-report delay
  ifelse simulation-speed <= 0
  [ report ln (10 / 0.001) ]
  [ report ln (10 / simulation-speed) ]
end

;; plot the histogram with the appropriate value
to do-histograms
  set-current-plot "Speed Histogram"
  set-current-plot-pen "speed"

  ;; on the last time thru, show the cumulative histogram
  if speed-index > num-speeds
  [ set cumulative-histogram? true ]

  ;; put all the turtles' speeds in speed-frequency-sum-list so that we can draw the cumulative histogram
  ask turtles
  [ set speed-frequency-sum-list lput speed speed-frequency-sum-list ]

  ;; if cumulative-histogram? is true, histogram all the values in all the speed-lists up to this point
  ifelse cumulative-histogram?
  [ histogram-list speed-frequency-sum-list ]
  [ histogram-from turtles [ speed ] ]
end

;; set certain variables to their appropriate intial values so that the same data can be run another time
;; also resets the histogram
to reset
  set clock 0
  set speed-index 0
  set speed-frequency-sum-list []
  set cumulative-histogram? false

  ask turtles
  [ setxy ((random-float (2 * screen-edge-x - 1)) - screen-edge-x + .5) ((random-float (2 * screen-edge-y - 1)) - screen-edge-y + .5) ]
  setup-histograms
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; initialize instructions to quickly setup the model, calculators, and TISchool webpage to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "First, run the physical People Molecules activity by..." "having your students simulate a gas by wearing CBRs set to collect..." 
    "speed information.  See Information for details on use of..." "CBR's and how to run the physical People Molecules activity."
    "Then, run the Navigator activity to collect the data:"
    "Students: If you used a CBR in the previous activity,..." "login to the calculator that you used."
    "A histogram of your speed will be displayed."
    "When finished viewing your histogram, you can view the class'..." "histogram by pressing ENTER followed by the VIEW softkey."
    "It may be necessary to press the RFSH button to get all the class' data."
    "Compare your speed distribution with the class' distribution."
    "Teacher: In the NetLogo activity, the class can view..." "the time evolution of the speed distribution:"
    "Optional- Zoom In (see Tools in the Menu Bar) - optional"
    "Press the GET CALC DATA button."
    "When all the students' have appeared in..." "the NetLogo Graphics Window, press the GO button to..." "play the students' speeds."
    "To replay the same data-set of speeds, stop the GO button by..." "pressing it again, if it is still depressed."
    "Press the RE-RUN button, and the GO button once again."
    
    "Teacher: To rerun the activity, do not clear the server."
    "Stop the simulation by pressing the NetLogo GO button."
    "Students: Collect new data with the CBR's."
    "Login to each calulator used to collect data."
    "Teacher: Press the GET CALC DATA button."
    "When all the students' have appeared in..." "the NetLogo Graphics Window, press the GO button to..." "play the students' speeds."
    
    "Teacher: To start the simulation over with a new group,..." "stop the model by pressing the NetLogo GO button, if it is on."
    "Have everyone, including yourself, logout of their calculators."
    "Press the ERASE ALL DATA button on the TISCHOOL site."
    "Press the NetLogo SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start item qs-item qs-items
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start item qs-item qs-items
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start item qs-item qs-items
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the calculators ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; gather data from the calculators, create turtles if necessary, and set some variables if there are turtles 
to get-student-data
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    exe-crt hubnet-message-source
  ]

  if any? turtles
  [
    set-color
    ask random-one-of turtles
    [ set num-speeds length speed-list ]
    set max-speed max values-from turtles [ max speed-list ]
  ]
end

;; only create a turtle with user-id equalling current-id if one doesn't already exist.
;; then initialize any turtles with user-id equalling current-id with the information from the server
to exe-crt [ current-id ]
  if( not any? turtles with [ user-id = current-id ] )
  [
    cct 1
    [ set user-id current-id ]
  ]
  ask turtles with [ user-id = current-id ]
  [ init-turtle-data ]
end

;; initialize turtles to appropriate values
to init-turtle-data
  locals
  [ index ]
  
  setxy ((random-float (2 * screen-edge-x - 1)) - screen-edge-x + .5) ((random-float (2 * screen-edge-y - 1)) - screen-edge-y + .5)
  rt random-float 360
  set index 0
  set speed-list []
  repeat length hubnet-message
  [
    set speed-list (lput (item index hubnet-message) speed-list)
    set index (index + 1)
  ]
end


; ***NetLogo Model Copyright Notice***

; This activity and associated models and materials was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
; The project gratefully acknowledges the support of the 
; National Science Foundation ( REPP program ) -- grant number REC #9814682.

; Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution for
; profit.

; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet People Molecules model.
; http://ccl.northwestern.edu/netlogo/models/HubNetPeopleMolecules.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.

; In other publications, please use:
; Copyright 1998 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetPeopleMolecules
; for terms of use.
;
; ***End NetLogo Model Copyright Notice***
@#$#@#$#@
GRAPHICS-WINDOW
236
94
671
550
12
12
17.0
1
12
1
1
1
0

CC-WINDOW
5
564
680
659
Command Center

BUTTON
7
10
69
43
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

BUTTON
126
10
230
43
Get Calc Data
get-student-data
NIL
1
T
OBSERVER
T
NIL

BUTTON
68
49
169
82
Re-Run
reset
NIL
1
T
OBSERVER
T
NIL

BUTTON
45
87
109
120
Go
go
T
1
T
OBSERVER
T
NIL

SLIDER
4
219
232
252
histogram-display-interval
histogram-display-interval
1
50
2
1
1
NIL

MONITOR
28
128
84
177
clock
clock
0
1

PLOT
1
294
235
515
Speed Histogram
speed
number
0.0
5.0
0.0
15.0
true
false
PENS
"speed" 1.0 1 -65536 true

MONITOR
243
10
642
59
Quick Start Instructions (more details in info tab)
quick-start
0
1

BUTTON
461
60
539
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

BUTTON
538
60
622
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
259
60
377
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

BUTTON
119
87
193
120
Go Once
go
NIL
1
T
OBSERVER
T
NIL

SWITCH
4
182
232
215
cumulative-histogram?
cumulative-histogram?
0
1
-1000

SLIDER
4
254
232
287
simulation-speed
simulation-speed
0
10
9.7
0.1
1
NIL

MONITOR
87
128
214
177
num of molecules
count turtles
3
1

@#$#@#$#@
WHAT IS IT?
-----------
Students emulate gas molecules and collect speed data using CBR's.  They can then aggregate this data and replay, in a limited sense, their actions.  They can see how the distribution of speeds changes in time, and can use this as a jumping off point for learning about the Maxwell Boltzmann distribution.

This activity requires the TI Navigator activity AAA - People Molecules 1.1

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------

QUICKSTART INSTRUCTIONS:
------------------------
Contains instructions as to how to quickly setup the model, calculators, and TISchool web page so as to run this activity.  The instructions can be found below:

IMPORTANT:  Follow the order described below.  If the students run the Navigator activity before collecting their speed data, the Navigator program will crash.

Teacher: First, run the physical People Molecules activity by having your students simulate a gas by wearing Calculator Based Rangers (CBR's) set to collect speed information.
Divide the class into two sections, one to play the part of a gas molecule and one to hold up poster boards or large sheets of paper (at least 3 ft x 3 ft).  The paper or poster boards are to reflect the sound waves emitted from the CBR.  There must be enough of the people holding poster boards to be able to form a square large enough for the people holding the CBR's to move around in.  Be sure to explain to each group what their role in the simulation is.  Have a brief discussion where the students discuss the rules they think the people who are the gas molecules should follow, and what they think they will see.  What sort of speed distribution do they expect?  If they were to take "snapshots" where they could know everyone's speed at one moment, what would they expect to find?

Everyone: If you are playing the part of a molecule, reset the RAM on your calculator (2nd MEM 7 1 2).  Connect a CBR to a calculator using a calculator-to-calculator or a calculator-to-CBR linking cable.  Download the program to use the CBR by:
1. On the calculator, going to the LINK Menu (by pressing 2nd followed by X,T,theta,n)
2. On the calculator, choosing the RECIEVE Menu (by pressing the RIGHT arrow key)
3. On the calculator, pressing ENTER
4. On the CBR, opening the flap that has the gold Sonic Sensor on it and pressing the 82/83 button
Once the RANGER program has been received successfully, run it by pressing the PRGM key and selecting the RANGER program.
prgmRANGER should appear on the Home screen of the calculator.  Press ENTER.
At the RANGER program's main menu, choose SETUP/SAMPLE.
Change the DISPLAY to VEL and then select START NOW.

If you are holding a sheet of paper or a poster board, form a square surrounding the people using the CBR's.  The square should be large enough for the people with the CBR's to be able to move freely in.

Once the square is formed around the people who are acting as gas molecules, the people who are simulating the gas molecules should hold the CBR to their chest, with the gold sound panel facing out horizontally.  Then they should press ENTER on the calculators and begin simulating gas molecules.

Once the physical simulation is done, the students quit the Ranger program by hitting ENTER (to exit the graph view) and then select the QUIT option from the RANGER Main Menu.  Information on general use of CBR's can be downloaded from the TI website at http://education.ti.com/product/tech/cbr/down/download.html and clicking on the "Getting Started with CBR" link.

Teacher: Now run the Navigator activity to collect the data:
Open the TISchool site in your web browser.
Enable activity AAA - People Molecules 1.1.
Open the teacher console and press the ERASE ALL DATA button.

Students who collected speed data with the CBR's: Login to the calculator that you used.
A histogram of your speed will be displayed.  When finished viewing your histogram, you can view the class' histogram by pressing ENTER followed by the VIEW softkey.  It may be necessary to press the RFSH button to get all the class' data.

Teacher: In the NetLogo activity, the class can view the time evolution of the speed distribution:
If it is not open already, open the NetLogo model.
If you are prompted by a Login dialog, enter your teacher id and password and any other necessary information.
Optional- Zoom In (see Tools in the Menu Bar)
Press the GET CALC DATA button.
When all the students' have appeared in the NetLogo Graphics Window, press the GO button to play the students' speeds.

To replay the same speeds, stop the GO button by pressing it again, if it is still depressed.
Press the RE-RUN button, and the GO button once again.

Teacher: To rerun the activity, do not clear the server.
Stop the simulation by pressing the NetLogo GO button.
Students: Collect new data with the CBR's.
Login to each calculator used to collect data.
Teacher: Press the GET CALC DATA button.
When all the students' have appeared in the NetLogo Graphics Window, press the GO button to play the students' speeds.

To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on.
Have everyone, including yourself, logout of their calculators.
Press the ERASE ALL DATA button on the TISCHOOL site.
Press the NetLogo SETUP button.
Follow these instructions from the beginning again.

BUTTONS:
--------
SETUP - initializes the model
GET CALC DATA - this gets the student speed data and creates gas molecules to reenact those speeds
GO - runs the simulation by having the gas molecules move at same speed as the students' speeds
GO ONCE - runs the simulation by having the gas molecules move thru the next speed in the list of the students' speeds
RE-RUN - resets the model to be able to run it again with the same data set of speeds
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

MONITORS:
---------
CLOCK - the number of times thru the GO procedure.  Every HISTOGRAM-DISPLAY-INTERVAL the histogram SPEED HISTOGRAM will update
NUM OF MOLECULES - displays the number of gas molecules in the Graphics Window

PLOTS:
-------------
SPEED HISTOGRAM - histograms the speeds of all the gas molecules based on the value of CUMULATIVE-HISTOGRAM? slider

SLIDERS:
--------
HISTOGRAM-DISPLAY-INTERVAL - the histogram will be updated every this many number of clock ticks
SIMULATION-SPEED - how fast the model runs

SWITCHES:
---------
CUMULATIVE-HISTOGRAM? - if this is true, the plot will histogram the sum of the all the speeds of the gas molecules up to this point.  If false, the plot will histogram the speeds of the gas molecules for this histogram interval


CALCULATOR INFORMATION
----------------------

TEACHER CALCULATOR:
-------------------
Students and teacher have identical calculator programs.

STUDENT CALCULATOR:
-------------------
The QUIT softkey prompts you to quit the calculator program.  Press the YES softkey if you really want to quit, and the NO softkey if you don't.
The RFSH softkey collects any additional information from the TI server.  It does nothing if there is no addition information.
The VIEW softkey toggles between the histogram of all the class's data and the histogram with just the information for this calculator.

In the histograms, you can find the values for each of the bars by pressing the RIGHT and LEFT arrow keys.  Press ENTER to return to the softkey Menu.


THINGS TO NOTICE
----------------
Where is the highest peek in the histogram?  The lowest peak?

What is the general shape of the histogram?

Do the individual histograms resemble the cumulative histogram?


THINGS TO TRY
-------------
When collecting the data in the physical People Molecules activity, challenge the class to do things like make the gas 
  -get warmer
  -get colder
  -increase the pressure of the gas
  -decrease the pressure of the gas
How should these different properties affect the speed histograms?  What needs to be done in order to make these properties happen?


EXTENDING THE MODEL
-------------------
Currently, the People Molecules activity only allows speed data to be collected.  Modify the NetLogo model to collect data on distance traveled, and acceleration in addition to speed.


CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the project:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
The project gratefully acknowledges the support of the National Science Foundation ( REPP program ) -- grant number REC #9814682.

Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

Permission to use, copy, or modify this software and its associated documentation, models and curricular materials for educational and research purposes only and without fee is hereby granted, provided that this copyright notice and the original authors' names appear on all copies and supporting documentation. For any other uses of this software, in original or modified form, including, but not limited to distribution in whole or in part, specific prior permission must be obtained from Uri Wilensky and Walter Stroup. These programs shall not be used, rewritten, or adapted as the basis of a commercial or hardware product without first obtaining appropriate licenses from Uri Wilensky & Walter Stroup. We make no representations about the suitability of this software for any purpose. It is provided "as is" without express or implied warranty.

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet People Molecules model. http://ccl.northwestern.edu/netlogo/models/HubNetPeopleMolecules. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1999 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetPeopleMolecules for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
